#include "string.h"
#include "math.h"

#include "windows.h"
#include "richedit.h"
#include "commdlg.h"

#include "resource.h"

#define GETTRUTH(a) a ? 'T' : 'F'
#define APPNAME "Truth Table Generator"

typedef struct
{
	char cVar;
	bool bVal;
} TABLE;

HWND hWndMain;
HWND hWndEdit;
HWND hWndRichEdit;
HWND hWndButton;

bool EvalExpression(char* szExpression, TABLE* tables, int numTables);
void EvalLeftRight(char* szExpression, int* iIndex, TABLE* tables, int numTables, bool* bLeft, bool* bRight);
void GenerateTruthTable(char* szInput, HWND hWnd);
int  GetSubExpressionLength(char* lpszSubExpression);
int  GetSubExpressionStart(char* lpszSubExpression, int iIndex);
void GetVariables(char* szInput, char* szVariables);
bool GetVarValue(char cVar, TABLE* tables, int numTables);
bool ParseInputString(char* szInput);
void Replace(char* szSource, char* szSearch, char* szReplace);

BOOL InitApplication(HINSTANCE hInstance);
BOOL InitInstance(HINSTANCE hInstance);
DWORD CALLBACK OutputFile(DWORD_PTR dwCookie, LPBYTE pbBuff, LONG cb, LONG *pcb);
void SaveFile();
LRESULT CALLBACK WndProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam);

class MalformedException
{
public:
	MalformedException(char* szException) {this->szException = szException;}
	char* szException;
};

void GenerateTruthTable(char* szInput, HWND hWnd)
{
	char szVariables[25];
	int iVariables;
	bool bRet;
	int iInputLength;
	int i;
	int iOutputPos = 0;
	int iOutputLength;

	bool bContinue = false;

	TABLE* tables;
	char* szResults;
	char* szParseString;
	
	bRet = ParseInputString(szInput);
	if (!bRet) return;
	
	GetVariables(szInput, szVariables);
	iVariables = strlen(szVariables);
	if (iVariables == 0) return;

	tables = new TABLE[iVariables];
	if (tables == NULL)
	{
		MessageBox(hWndMain, "There is not enough memory available.", APPNAME, MB_ICONEXCLAMATION);
		return;
	}

	for(i = 0; i < iVariables; i++)
	{
		tables[i].bVal = false;
		tables[i].cVar = szVariables[i];
	}

	iInputLength = strlen(szInput)+1;
	iOutputLength = static_cast<int>((pow(2, iVariables) + 1) * (2 * iVariables + 3) + 4);
	szParseString = new char[iInputLength];
	szResults = new char[iOutputLength];
	if ((szParseString == NULL) || (szResults == NULL))
	{
		MessageBox(hWndMain, "There is not enough memory available.", APPNAME, MB_ICONEXCLAMATION);
		delete [] tables;
		delete [] szParseString;
		delete [] szResults;
		return;
	}

	for(i = 0; i < iVariables; i++)
	{
		szResults[iOutputPos++] = tables[i].cVar;
		szResults[iOutputPos++] = ' ';
	}

	szResults[iOutputPos] = '\0';
	strcat(szResults, "Result");
	iOutputPos += 6;
		
	do
	{
		strcpy(szParseString, szInput);
		try
		{
			bRet = EvalExpression(szParseString, tables, iVariables);
		}
		catch (MalformedException e)
		{
			MessageBox(hWndMain, e.szException, APPNAME, MB_ICONEXCLAMATION);
			szResults[0] = '\0';
			break;
		}

		szResults[iOutputPos++] = 13;
		szResults[iOutputPos++] = 10;

		for(i = 0; i < iVariables; i++)
		{
			szResults[iOutputPos++] = GETTRUTH(tables[i].bVal);
			szResults[iOutputPos++] = ' ';
		}

		szResults[iOutputPos++] = GETTRUTH(bRet);

		bContinue = false;

		for(i = 0; i < iVariables; i++)
		{
			tables[i].bVal = !tables[i].bVal;
			if (tables[i].bVal)
			{
				bContinue = true;
				break;
			}
		}

	} while (bContinue);

	delete [] tables;
	delete [] szParseString;

	szResults[iOutputPos] = '\0';
	SendMessage(hWnd, WM_SETTEXT, 0, reinterpret_cast<LPARAM>(szResults));
	delete [] szResults;
}

bool EvalExpression(char* szExpression, TABLE* tables, int numTables)
{
	int iIndex = 0;

	while (szExpression[iIndex] != 0)
	{
		if (szExpression[iIndex] == '!')
		{
			int i = iIndex+1;
			int j;
			char cNext = szExpression[iIndex+1];
			switch (cNext) {
				case '(':
				{
					int iSubExpressionLength = GetSubExpressionLength(szExpression+iIndex+1);
					char *szSubExpression = new char[iSubExpressionLength+1];
					if (szSubExpression == NULL)
					{
						throw MalformedException("There is not enough memory available.");
					}

					bool bResult;
					strncpy(szSubExpression, szExpression+iIndex+2, iSubExpressionLength);
					szSubExpression[iSubExpressionLength] = '\0';

					try
					{
						bResult = !EvalExpression(szSubExpression, tables, numTables);
					}
					catch (MalformedException e)
					{
						delete [] szSubExpression;
						throw e;
					}

					szExpression[iIndex] = GETTRUTH(bResult);
					delete [] szSubExpression;

					j = i+iSubExpressionLength+2;
					break;
				}
				default:
					
					cNext &= 0xDF; // Convert to UCASE if letter
					if (cNext >= 65 && cNext <= 91)
					{
						bool bResult = !GetVarValue(cNext, tables, numTables);
						szExpression[iIndex] = GETTRUTH(bResult);

						j = iIndex+2;
					}
					else
					{
						throw MalformedException("Not must be followed by an expression.");
					}
					break;
			}

			do
			{
				szExpression[i] = szExpression[j];
				i++;
				j++;
			} while (szExpression[i-1] != '\0');
		}
		iIndex++;
	}

	iIndex = 0;
	while (szExpression[iIndex] != 0)
	{
		if (szExpression[iIndex] == '&')
		{
			bool bLeft;
			bool bRight;
			EvalLeftRight(szExpression, &iIndex, tables, numTables, &bLeft, &bRight);
			bool bResult = bLeft && bRight;

			szExpression[iIndex-1] = GETTRUTH(bResult);

			int i = iIndex;
			do
			{
				szExpression[i] = szExpression[i+2];
				i++;
			} while (szExpression[i-1] != '\0');
		}
		else
		{
			iIndex++;
		}
	}

	iIndex = 0;
	while (szExpression[iIndex] != 0)
	{
		if ((szExpression[iIndex] == '|') || (szExpression[iIndex] == '^'))
		{
			bool bLeft;
			bool bRight;
			EvalLeftRight(szExpression, &iIndex, tables, numTables, &bLeft, &bRight);

			bool bResult;
			if (szExpression[iIndex] == '|')
			{
				bResult = bLeft || bRight;
			}
			else
			{
				bResult = bLeft ^ bRight;
			}

			szExpression[iIndex-1] = GETTRUTH(bResult);

			int i = iIndex;
			do
			{
				szExpression[i] = szExpression[i+2];
				i++;
			} while (szExpression[i-1] != '\0');
		}
		else
		{
			iIndex++;
		}
	}

	iIndex = 0;
	while (szExpression[iIndex] != 0)
	{
		if (szExpression[iIndex] == '>')
		{
			bool bLeft;
			bool bRight;
			EvalLeftRight(szExpression, &iIndex, tables, numTables, &bLeft, &bRight);

			bool bResult = (!bLeft) || bRight;

			szExpression[iIndex-1] = GETTRUTH(bResult);

			int i = iIndex;
			do
			{
				szExpression[i] = szExpression[i+2];
				i++;
			} while (szExpression[i-1] != '\0');
		}
		else
		{
			iIndex++;
		}
	}

	iIndex = 0;
	while (szExpression[iIndex] != 0)
	{
		if (szExpression[iIndex] == '=')
		{
			bool bLeft;
			bool bRight;
			EvalLeftRight(szExpression, &iIndex, tables, numTables, &bLeft, &bRight);

			bool bResult = (bLeft == bRight);

			szExpression[iIndex-1] = GETTRUTH(bResult);

			int i = iIndex;
			do
			{
				szExpression[i] = szExpression[i+2];
				i++;
			} while (szExpression[i-1] != '\0');
		}
		else
		{
			iIndex++;
		}
	}

	iIndex = 0;
	char cResult = '\0';
	int iNumBracketsOpen = 0;
	int iNumBracketsClose = 0;

	while (szExpression[iIndex] != '\0')
	{
		switch(szExpression[iIndex]) {
			case '(':
				iNumBracketsOpen++;
				if (cResult != 0)
				{
					throw MalformedException("All brackets must contain an expression.");
				}
				break;
			case ')':
				iNumBracketsClose++;
				if (cResult == 0)
				{
					throw MalformedException("All brackets must contain an expression.");
				}
				break;
			default:
				if (cResult == 0)
				{
					cResult = szExpression[iIndex];
				}
				else
				{
					throw MalformedException("All variables must be seperated by an operator.");
				}
		}
		iIndex++;
	}

	if (iNumBracketsOpen < iNumBracketsClose)
	{
		throw MalformedException("All brackets must be opened.");
	}
	else if (iNumBracketsOpen > iNumBracketsClose)
	{
		throw MalformedException("All brackets must be closed.");
	}
	else if (cResult == 0)
	{
		throw MalformedException("All brackets must contain an expression.");
	}
	else
	{
		return GetVarValue(cResult, tables, numTables);
	}
}

int GetSubExpressionLength(char* lpszSubExpression)
{
	int i = 0;
	int iDepth = 0;
	do
	{
		if (lpszSubExpression[i] == '(')
		{
			iDepth++;
		}
		else if (lpszSubExpression[i] == ')')
		{
			iDepth--;
		}
		else if (lpszSubExpression[i] == '\0')
		{
			throw MalformedException("All brackets must be closed.");
		}
		i++;
	} while (iDepth != 0);

	return i-2;
}

int GetSubExpressionStart(char* lpszSubExpression, int iIndex)
{
	int i = iIndex;
	int iDepth = 0;
	do
	{
		if (lpszSubExpression[i] == '(')
		{
			iDepth--;
		}
		else if (lpszSubExpression[i] == ')')
		{
			iDepth++;
		}
		i--;
	} while ((iDepth != 0) && (i != -1));

	if ((i == -1) && (iDepth != 0))
	{
		throw MalformedException("All brackets must be opened.");
	}

	return i+1;
}

bool GetVarValue(char cVar, TABLE* tables, int numTables)
{
	if (cVar == 'F')
	{
		return false;
	}
	else if (cVar == 'T')
	{
		return true;
	}
	else
	{
		for(int i = 0; i < numTables; i++)
		{
			if (tables[i].cVar == cVar) // Convert both to UCASE
			{
				return tables[i].bVal;
			}
		}
		throw MalformedException("Variable not found.");
	}
}

void EvalLeftRight(char* szExpression, int* iIndex, TABLE* tables, int numTables, bool* bLeft, bool* bRight)
{
	if (*iIndex == 0)
	{
		throw MalformedException("All binary operators must have a left operand.");
	}
	else if (szExpression[*iIndex+1] == '\0')
	{
		throw MalformedException("All binary operators must have a right operand.");
	}

	if (szExpression[*iIndex-1] == ')')
	{
		int iStart = GetSubExpressionStart(szExpression, *iIndex - 1);
		int iLength = *iIndex-iStart-2;
		char* szSubExpression = new char[iLength+1];
		if (szSubExpression == NULL)
		{
			throw MalformedException("There is not enough memory available.");
		}

		strncpy(szSubExpression, szExpression+iStart+1, iLength);
		szSubExpression[iLength] = '\0';

		try
		{
			*bLeft = EvalExpression(szSubExpression, tables, numTables);
		}
		catch (MalformedException e)
		{
			delete [] szSubExpression;
			throw e;
		}
		delete [] szSubExpression;

		int i = iStart+1;
		int j = *iIndex;
		do
		{
			szExpression[i] = szExpression[j];
			i++;
			j++;
		} while (szExpression[i-1] != '\0');

		*iIndex = iStart+1;
	}
	else if (((szExpression[*iIndex-1] &= 0xDF) >= 65) && (szExpression[*iIndex-1] <= 91))
	{
		*bLeft = GetVarValue(szExpression[*iIndex-1], tables, numTables);
	}
	else
	{
		throw MalformedException("All binary operators must have a left operand.");
	}

	if (szExpression[*iIndex+1] == '(')
	{
		int iLength = GetSubExpressionLength(szExpression+*iIndex+1);
		char* szSubExpression = new char[iLength+1];
		if (szSubExpression == NULL)
		{
			throw MalformedException("There is not enough memory available.");
		}

		strncpy(szSubExpression, szExpression+*iIndex+2, iLength);
		szSubExpression[iLength] = '\0';

		try
		{
			*bRight = EvalExpression(szSubExpression, tables, numTables);
		}
		catch (MalformedException e)
		{
			delete [] szSubExpression;
			throw e;
		}
		delete [] szSubExpression;

		int i = *iIndex+2;
		int j = *iIndex + iLength + 3;
		do
		{
			szExpression[i] = szExpression[j];
			i++;
			j++;
		} while (szExpression[i-1] != '\0');
	}
	else if (((szExpression[*iIndex+1] &= 0xDF) >= 65) && (szExpression[*iIndex+1] <= 91))
	{
		*bRight = GetVarValue(szExpression[*iIndex+1], tables, numTables);
	}
	else
	{
		throw MalformedException("All binary operators must have a right operand.");
	}
}

bool ParseInputString(char* szInput)
{
	strupr(szInput);

    Replace(szInput, "<=>", "=");
    Replace(szInput, "<==>", "=");
    Replace(szInput, "==>", ">");
    Replace(szInput, "=>", ">");
    Replace(szInput, "!!", "");
    
    Replace(szInput, "AND", "&");
    Replace(szInput, "XOR", "^");
    Replace(szInput, "OR", "|");
    Replace(szInput, "NOT", "!");
    Replace(szInput, "EQUALS", "=");
    Replace(szInput, "IMPLIES", ">");

	Replace(szInput, " ", "");

	for(int i = 0; szInput[i] != '\0'; i++)
	{
        if ((szInput[i] != 61) && (szInput[i] != 62) && (szInput[i] != 33) &&
			(szInput[i] != 38) && (szInput[i] != 40) && (szInput[i] != 41) &&
			(szInput[i] != 124) && (szInput[i] != 94) &&
			((szInput[i] < 65 || (szInput[i] > 91))))
		{
			char szMessage[] = "Invalid character: \0\0";
			szMessage[strlen(szMessage)] = szInput[i];
			MessageBox(0, szMessage, APPNAME, MB_ICONEXCLAMATION);
			return false;
		}
	}

	return true;
}

void Replace(char* szSource, char* szSearch, char* szReplace)
{
	int iSearchLen = strlen(szSearch);
	int iReplaceLen = strlen(szReplace);

	char* szNextLoc = szSource;
	char* szFound;
	char* szCurLoc = szSource;

	while (true)
	{
		szFound = strstr(szNextLoc, szSearch);
		if (szFound == 0) break;

		if (szNextLoc != szCurLoc)
		{
			while (szNextLoc < szFound)
			{
				*szCurLoc = *szNextLoc;
				szCurLoc++;
				szNextLoc++;
			}
		}
		else
		{
			szCurLoc = szFound;
			szNextLoc = szFound;
		}

		strcpy(szCurLoc, szReplace);
		szCurLoc += iReplaceLen;
		szNextLoc += iSearchLen;
	}

	if (szNextLoc != szCurLoc)
	{
		do
		{
			*szCurLoc = *szNextLoc;
			szCurLoc++;
		} while (*szNextLoc++ != '\0');
	}
}

void GetVariables(char* szInput, char* szVariables)
{
	int j = 0;
	szVariables[0] = '\0';
	for(int i = 0; szInput[i] != '\0'; i++)
	{
		if ((szInput[i] >= 65) && (szInput[i] <= 91) && (szInput[i] != 'T') && (szInput[i] != 'F'))
		{
			if (strchr(szVariables, szInput[i]) == NULL)
			{
				szVariables[j] = szInput[i];
				j++;
			}
		}
	}
	szVariables[j] = '\0';
}

LRESULT CALLBACK WndProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
		case WM_CLOSE:
			DestroyWindow(hWnd);
			return FALSE;
			break;
		case WM_DESTROY:
			PostQuitMessage(1);
			return FALSE;
			break;
		case WM_HELP:
			WinHelp(hWnd, "TruthTable.hlp", HELP_CONTEXT, 1);
			return TRUE;
			break;
		case WM_INITMENU:
			LPARAM iLength;
			iLength = SendMessage(hWndRichEdit, WM_GETTEXTLENGTH, 0, 0);
			if (iLength>0)
			{
				EnableMenuItem(reinterpret_cast<HMENU>(wParam), IDM_SAVE, MF_ENABLED);
			}
			else
			{
				EnableMenuItem(reinterpret_cast<HMENU>(wParam), IDM_SAVE, MF_GRAYED);
			}

			return FALSE;
			break;
		case WM_COMMAND:
			switch (LOWORD(wParam))
			{
				case 1: // Generate
					char *szInput;
					LPARAM iLength;

					iLength = SendMessage(hWndEdit, WM_GETTEXTLENGTH, 0, 0)+1;
					szInput = new char[iLength];
					if (szInput == NULL)
					{
						MessageBox(hWndMain, "There is not enough memory available.", APPNAME, MB_ICONEXCLAMATION);
						return TRUE;
					}
					SendMessage(hWndEdit, WM_GETTEXT, iLength, reinterpret_cast<LPARAM>(szInput));

					GenerateTruthTable(szInput, hWndRichEdit);
					delete [] szInput;
					break;
				case IDM_CLOSE:
					DestroyWindow(hWnd);
					break;
				case IDM_SAVE:
					SaveFile();
					break;
				case ID_LOGICOPERATORS:
					WinHelp(hWnd, "TruthTable.hlp", HELP_CONTEXT, 1);
					break;
				case IDM_ABOUT:
					MessageBox(hWnd, "This program is freeware and was written by Jason Bouzane.\n\nThanks to Omar Ismail for the idea.", APPNAME, MB_ICONINFORMATION);
					break;
			}


			return FALSE;
			break;
		case WM_NOTIFY:
			if (wParam == 2)
			{
				MSGFILTER *msg;
				msg = reinterpret_cast<MSGFILTER *>(lParam);
				if ((msg->msg == WM_KEYDOWN) && (msg->wParam == 9))
				{
					DWORD dwRet = GetKeyState(VK_SHIFT);
					HWND hWndNext = GetNextDlgTabItem(hWnd, msg->nmhdr.hwndFrom, dwRet & 0x80000000);
					SetFocus(hWndNext);
					return 1;
				}
				return FALSE;
				break;
			}
			return TRUE;
			break;
		case WM_SIZE:
			if (wParam != SIZE_MINIMIZED)
			{
				int iWidth = LOWORD(lParam);
				int iHeight = HIWORD(lParam);

				MoveWindow(hWndEdit, 6, 6, iWidth - 99, 21,  TRUE);
				MoveWindow(hWndButton, iWidth - 87, 4, 81, 25, TRUE);
				MoveWindow(hWndRichEdit, 6, 35, iWidth - 12, iHeight - 41, TRUE);
			}

			return FALSE;
			break;
		case WM_GETMINMAXINFO:
			MINMAXINFO* mmi;
			mmi = reinterpret_cast<MINMAXINFO *>(lParam);
			mmi->ptMinTrackSize.x = 200;
			mmi->ptMinTrackSize.y = 250;
			return FALSE;
			break;
		default:
			return DefWindowProc(hWnd, uMsg, wParam, lParam);
			break;
	}
}

BOOL InitApplication(HINSTANCE hInstance)
{
	LoadLibrary("RichEd32.Dll");
	WNDCLASS wc;
	ZeroMemory(&wc, sizeof(wc));
	wc.lpfnWndProc = &WndProc;
	wc.hInstance = hInstance;
	wc.lpszClassName = "MAINCLASS";
	wc.hbrBackground = (HBRUSH) COLOR_WINDOW;
	wc.hCursor = LoadCursor(NULL, MAKEINTRESOURCE(IDC_ARROW));
	wc.hIcon = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_MAIN));
	wc.lpszMenuName = MAKEINTRESOURCE(IDR_MAINMENU);
	return (RegisterClass(&wc) != 0);
}

BOOL InitInstance(HINSTANCE hInstance)
{
	HFONT hNormalFont = reinterpret_cast<HFONT>(GetStockObject(DEFAULT_GUI_FONT));
	HDC hDC = GetDC(hWndMain);
	
	int nHeight = -MulDiv(9, GetDeviceCaps(hDC, LOGPIXELSY), 72);
	HFONT hTextFont = CreateFont(nHeight, 0, 0, 0, 0, FALSE, FALSE, FALSE, ANSI_CHARSET, OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, FIXED_PITCH  | FF_DONTCARE, "Fixedsys");
	
	hWndMain = CreateWindowEx(0, "MAINCLASS", APPNAME, WS_OVERLAPPEDWINDOW, 0,0, 313, 371, NULL, NULL, hInstance, NULL);
	if (hWndMain == 0) return FALSE;

	hWndEdit = CreateWindowEx(WS_EX_OVERLAPPEDWINDOW, "EDIT", NULL, WS_BORDER | WS_CHILD | WS_VISIBLE | WS_TABSTOP | ES_AUTOHSCROLL, 6, 6, 201, 21, hWndMain, NULL, hInstance, NULL);
	hWndButton = CreateWindow("BUTTON", "&Generate", BS_PUSHBUTTON | BS_DEFPUSHBUTTON | WS_CHILD | WS_VISIBLE | WS_TABSTOP, 216, 4, 81, 25, hWndMain, (HMENU)1, hInstance, NULL);
	hWndRichEdit = CreateWindowEx(WS_EX_RIGHTSCROLLBAR, "RichEdit", NULL, ES_AUTOVSCROLL | ES_AUTOHSCROLL | ES_READONLY | ES_MULTILINE | WS_BORDER | WS_CHILD | WS_VISIBLE | WS_TABSTOP, 6, 40, 289, 265, hWndMain, (HMENU)2, hInstance, NULL);

	SendMessage(hWndEdit, WM_SETFONT, reinterpret_cast<WPARAM>(hTextFont), TRUE);
	SendMessage(hWndButton, WM_SETFONT, reinterpret_cast<WPARAM>(hNormalFont), TRUE);
	SendMessage(hWndRichEdit, WM_SETFONT, reinterpret_cast<WPARAM>(hTextFont), TRUE);

	SendMessage(hWndRichEdit, EM_SETEVENTMASK, 0, ENM_KEYEVENTS);
	SendMessage(hWndRichEdit, EM_SHOWSCROLLBAR, SB_VERT | SB_HORZ, TRUE);
	SendMessage(hWndEdit, EM_LIMITTEXT, 1000, 0);

	SetWindowPos(hWndButton, hWndEdit, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE);
	SetWindowPos(hWndRichEdit, hWndButton, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE);

	SetFocus(hWndEdit);
	ShowWindow(hWndMain, SW_SHOW);
	return TRUE;
}

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nShowCmd)
{
	if (!InitApplication(hInstance)) return 0;
	if (!InitInstance(hInstance)) return 0;

	HACCEL hAccel = LoadAccelerators(hInstance, MAKEINTRESOURCE(IDR_ACCEL));
	while (true)
	{
		MSG msg;
		BOOL bContinue = GetMessage(&msg, 0, 0, 0);
		if ((bContinue == -1)  || (!bContinue)) break;

		if (!TranslateAccelerator(hWndMain, hAccel, &msg))
		{
			if (!IsDialogMessage(hWndMain, &msg))
			{
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
		}
	}

	return 1;
}

void SaveFile()
{
	TCHAR szFilename[MAX_PATH+1] = "Output1.txt\0";
	OPENFILENAME ofn;

	ZeroMemory(&ofn, sizeof(ofn));
	ofn.Flags = OFN_ENABLESIZING | OFN_HIDEREADONLY | OFN_NOREADONLYRETURN | OFN_OVERWRITEPROMPT;
	ofn.hwndOwner = hWndMain;
	ofn.lpstrDefExt = "txt";
	ofn.lpstrInitialDir = NULL;
	ofn.lpstrFile = szFilename;
	ofn.lpstrFilter = "Text Files (*.txt)\0*.txt\0All Files (*.*)\0*.*\0\0";
	ofn.lpstrTitle = "Save Output File...";
	ofn.lStructSize = sizeof(ofn);
	ofn.nFilterIndex = 0;
	ofn.nMaxFile = MAX_PATH+1;

	if (GetSaveFileName(&ofn))
	{
		HANDLE hFile = CreateFile(szFilename, GENERIC_WRITE, FILE_SHARE_READ, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
		if (hFile == INVALID_HANDLE_VALUE)
		{
			MessageBox(hWndMain, "There was an error writing to the file.", APPNAME, MB_ICONEXCLAMATION);
		}
		else
		{
			EDITSTREAM es;
			es.dwCookie = reinterpret_cast<DWORD_PTR>(hFile);
			es.pfnCallback = &OutputFile;
			es.dwError = 0;
			SendMessage(hWndRichEdit, EM_STREAMOUT, SF_TEXT, reinterpret_cast<LPARAM>(&es));
			CloseHandle(hFile);
		}
	}
}

DWORD CALLBACK OutputFile(DWORD dwCookie, LPBYTE pbBuff, LONG cb, LONG *pcb)
{
	HANDLE hFile = reinterpret_cast<HANDLE>(dwCookie);
	return WriteFile(hFile, pbBuff, cb, reinterpret_cast<ULONG *>(pcb), NULL) == 0;
}